#!/bin/bash

cd "$(dirname "${BASH_SOURCE[0]}")"

EXITCODE=0

ESC_SEQ="\033["
COL_RESET=$ESC_SEQ"39;49;00m"
COL_RED=$ESC_SEQ"31;01m"
COL_GREEN=$ESC_SEQ"32;01m"
COL_YELLOW=$ESC_SEQ"33;01m"
COL_BLUE=$ESC_SEQ"34;01m"
COL_MAGENTA=$ESC_SEQ"35;01m"
COL_CYAN=$ESC_SEQ"36;01m"
COL_BOLD=$ESC_SEQ"1m"

echo ""
echo -e "${COL_BOLD}Checking to see if RoonServer can run on this machine ${COL_RESET}"
echo ""

function display_check  {
    printf "    %-60s" "$1"
}

function display_ok {
    echo -e " [   ${COL_GREEN}OK${COL_RESET}   ]"
}

function display_failed {
    echo -e " [ ${COL_RED}FAILED${COL_RESET} ]"
    EXITCODE=1
}

function check_ffmpeg_avconv {
    display_check "Checking for ffmpeg or avconv"
    which ffmpeg >/dev/null; WHICH_FFMPEG=$?
    which avconv >/dev/null; WHICH_AVCONV=$?
    if [ $WHICH_FFMPEG = 0 -o $WHICH_AVCONV = 0 ]; then
        display_ok
    else
        display_failed
    fi
}

function check_alsa {
    display_check "Checking for ALSA Libraries"
    Appliance/check_alsa >/dev/null 2>&1
    ALSA_RESULT=$?
    if [ x$ALSA_RESULT = x0 ]; then
        display_ok
    else
        display_failed
    fi
}

function check_bincompat {
    display_check "Checking for Binary Compatibility"
    Appliance/check_bincompat >/dev/null 2>&1
    BINCOMPAT_RESULT=$?
    if [ x$BINCOMPAT_RESULT = x0 ]; then
        display_ok
    else
        display_failed
    fi
}

function check_ulimit {
    display_check "Testing ulimit -n 8192"
    ulimit -n 8192 > /dev/null 2>&1
    ULIMIT_RESULT=$?
    if [ x$ULIMIT_RESULT = x0 ]; then
        display_ok
    else
        display_failed
    fi
}

function check_mount_cifs {
    display_check "Checking for the mount.cifs command"
    which mount.cifs >/dev/null; WHICH_MOUNT_CIFS=$?
    if [ $WHICH_MOUNT_CIFS = 0 ]; then
        display_ok
    else
        display_failed
    fi
}

function check_libicu {
    display_check "Checking for libicu"
    which icuinfo >/dev/null; WHICH_ICUINFO=$?
    ldconfig -p | grep libicu >/dev/null; LDCONFIG_LIBICU=$?
    if [ $WHICH_ICUINFO = 0 -o $LDCONFIG_LIBICU = 0 ]; then
        display_ok
    else
        display_failed
    fi
}

check_bincompat
check_alsa
check_ffmpeg_avconv
check_mount_cifs
check_libicu
check_ulimit

if [ $EXITCODE = 0 ]; then
    echo ""
    echo -e "${COL_BOLD}STATUS: ${COL_GREEN}SUCCESS${COL_RESET}"
    echo ""
else
    echo ""
    echo -e "STATUS: ${COL_RED}FAILED${COL_RESET}"
    echo ""
    echo "These issues must be addressed before RoonServer will run on this machine."
    echo ""
    echo "For more information on how to address this, see:"
    echo ""
    echo "   http://kb.roonlabs.com/LinuxInstall"
    echo ""
fi

exit $EXITCODE
